/**************************************************************************
 *
 * Copyright 2010, 2011 BMW Car IT GmbH
 * Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 * Copyright (c) 2013 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <EGL/egl.h>
#include <GLES2/gl2.h>

#include "WLEGLSurface.h"
#include "SubdivisionRenderer.h"
#include "Subdivision.h"

#define WL_UNUSED(A) (A)=(A)

extern int gNeedRedraw;

using namespace LayerManagerCalibration;

//////////////////////////////////////////////////////////////////////////////

typedef struct _es2shaderobject {
    GLuint fragmentShaderId;
    GLuint vertexShaderId;
    GLuint shaderProgramId;
    GLint  matrixLocation;
    GLint  colorLocation;
} ES2ShaderObject;

typedef struct _es2vertexbufferobject {
    GLuint vbo;
} ES2VertexBuffer;

static ES2ShaderObject	gShaderObject;
static ES2VertexBuffer	gVertexBuffer;

// Fragment shader code
const char* sourceFragShader = "  \
    uniform mediump vec4 u_color; \
    void main(void)               \
    {                             \
        gl_FragColor = u_color;   \
    }";

// Vertex shader code
const char* sourceVertexShader = " \
    attribute highp vec4 a_vertex; \
    uniform mediump mat4 u_matrix; \
    void main(void)                \
    {                              \
        gl_Position = u_matrix * a_vertex; \
    }";

//////////////////////////////////////////////////////////////////////////////

static bool InitShader();
static bool InitVertexBuffer(InputDeviceConfiguration* conf, float hScale, float vScale);


float world_mtx[16] = { 1.0,  0.0,  0.0,  0.0,
                        0.0,  1.0,  0.0,  0.0,
                        0.0,  0.0,  1.0,  0.0,
                       -1.0, -1.0,  0.0,  1.0};

bool
InitRenderer(InputDeviceConfiguration* conf,
             unsigned int dispWidth,
             unsigned int dispHeight,
             unsigned int touchWidth,
             unsigned int touchHeight)
{
    glViewport(0, 0, dispWidth, dispHeight);

    float hScale = (float) dispWidth / (float) touchWidth;
    float vScale = (float) dispHeight / (float) touchHeight;

    if (!InitShader()){
        return false;
    }

    if (!InitVertexBuffer(conf, hScale, vScale)){
        return false;
    }

    world_mtx[0] = 2.0/dispWidth;
    world_mtx[5] = 2.0/dispHeight;
    world_mtx[12] = -(float) dispWidth * world_mtx[0] / 2.0;
    world_mtx[13] = (float) dispHeight * world_mtx[5] / 2.0;

    glClearColor(0.2, 0.2, 0.2, 1.0);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);
    return true;
}


void
TerminateRenderer()
{
    glDeleteProgram(gShaderObject.shaderProgramId);
    glDeleteShader(gShaderObject.fragmentShaderId);
    glDeleteShader(gShaderObject.vertexShaderId);
}

char*
BuildShaderErrorLog(GLuint id)
{
    int l, nChar;
    glGetShaderiv(id, GL_INFO_LOG_LENGTH, &l);

    char* info = (char*)malloc(sizeof(char) * l);
    glGetShaderInfoLog(id, l, &nChar, info);

    return info;
}

static bool
InitShader()
{
    GLint result = 0;
    char* log = NULL;

    // Create the fragment shader object
    gShaderObject.fragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
    glShaderSource(gShaderObject.fragmentShaderId, 1,
        (const char**)&sourceFragShader, NULL);
    glCompileShader(gShaderObject.fragmentShaderId);

    glGetShaderiv(gShaderObject.fragmentShaderId, GL_COMPILE_STATUS, &result);
    if (!result){
        log = BuildShaderErrorLog(gShaderObject.fragmentShaderId);
        fprintf(stderr, "Failed to compile fragment shader: %s\n", log);
        if (log) free(log);
        return false;
    }

    // Create the vertex shader object
    gShaderObject.vertexShaderId = glCreateShader(GL_VERTEX_SHADER);
    glShaderSource(gShaderObject.vertexShaderId, 1,
        (const char**)&sourceVertexShader, NULL);
    glCompileShader(gShaderObject.vertexShaderId);

    glGetShaderiv(gShaderObject.vertexShaderId, GL_COMPILE_STATUS, &result);
    if (!result){
        log = BuildShaderErrorLog(gShaderObject.vertexShaderId);
        fprintf(stderr, "Failed to compile fragment shader: %s\n", log);
        if (log) free(log);
        return false;
    }

    gShaderObject.shaderProgramId = glCreateProgram();

    glAttachShader(gShaderObject.shaderProgramId, gShaderObject.fragmentShaderId);
    glAttachShader(gShaderObject.shaderProgramId, gShaderObject.vertexShaderId);

    glBindAttribLocation(gShaderObject.shaderProgramId, 0, "a_vertex");

    glLinkProgram(gShaderObject.shaderProgramId);

    glGetProgramiv(gShaderObject.shaderProgramId, GL_LINK_STATUS, &result);
    if (!result){
        log = BuildShaderErrorLog(gShaderObject.shaderProgramId);
        fprintf(stderr, "Failed to compile fragment shader: %s\n", log);
        if (log) free(log);
        return false;
    }

    glUseProgram(gShaderObject.shaderProgramId);
    gShaderObject.matrixLocation = glGetUniformLocation(
        gShaderObject.shaderProgramId, "u_matrix");
    gShaderObject.colorLocation = glGetUniformLocation(
        gShaderObject.shaderProgramId, "u_color");

    return true;
}

static bool
InitVertexBuffer(InputDeviceConfiguration* conf, float hScale, float vScale)
{
    glGenBuffers(1, &gVertexBuffer.vbo);
    glBindBuffer(GL_ARRAY_BUFFER, gVertexBuffer.vbo);

    list<Subdivision*> subdivisions = conf->getSubdivisions();
    const int vertsPerSub = 4;
    const int valsPerVertex = 2;
    int data_size = subdivisions.size() * vertsPerSub * valsPerVertex;

    printf("Scaling %f x %f\n", hScale, vScale);

    printf("Found %d subdivisions\n", subdivisions.size());

    GLfloat* data = new GLfloat[data_size];

    int subNum = 0;
    list<Subdivision*>::reverse_iterator iter;
    for (iter = subdivisions.rbegin();
         iter != subdivisions.rend();
         iter++)
    {
        coordinate topLeft = (*iter)->getTopLeftCoordinate();
        coordinate bottomRight = (*iter)->getBottomRightCoordinate();

        coordinate bottomLeft = {
            topLeft.x,
            bottomRight.y
        };

        coordinate topRight = {
            bottomRight.x,
            topLeft.y
        };

        int indexStart = subNum * vertsPerSub * valsPerVertex;

        // Apply calibration transformation to subdivisions
        if ((*iter)->getUncalibrated() == false)
        {
            conf->getCalibration().inverseTransformCoordinate(topLeft, topLeft);
            conf->getCalibration().inverseTransformCoordinate(bottomRight, bottomRight);
            conf->getCalibration().inverseTransformCoordinate(bottomLeft, bottomLeft);
            conf->getCalibration().inverseTransformCoordinate(topRight, topRight);
        }

        // vertical scaling is negative because +y points up for rendering,
        // but down for touch coordinates

        printf("Subdivision %d\n", subNum);
        data[indexStart + 0] = (GLfloat) bottomLeft.x   * hScale;
        data[indexStart + 1] = (GLfloat) bottomLeft.y   * -vScale;

        data[indexStart + 2] = (GLfloat) bottomRight.x  * hScale;
        data[indexStart + 3] = (GLfloat) bottomRight.y  * -vScale;

        data[indexStart + 4] = (GLfloat) topRight.x     * hScale;
        data[indexStart + 5] = (GLfloat) topRight.y     * -vScale;

        data[indexStart + 6] = (GLfloat) topLeft.x      * hScale;
        data[indexStart + 7] = (GLfloat) topLeft.y      * -vScale;

        printf("%f,%f  %f,%f  %f,%f  %f,%f\n",
               data[indexStart + 0],
               data[indexStart + 1],
               data[indexStart + 2],
               data[indexStart + 3],
               data[indexStart + 4],
               data[indexStart + 5],
               data[indexStart + 6],
               data[indexStart + 7]);

        subNum++;
    }

    glBufferData(GL_ARRAY_BUFFER, sizeof(GLfloat) * data_size, data,
                 GL_STATIC_DRAW);

    delete [] data;

    return true;
}

static void 
AttachVertexBuffer()
{
    glBindBuffer(GL_ARRAY_BUFFER, gVertexBuffer.vbo);
    glEnableVertexAttribArray(0);
    glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, 0);
}

static void 
DetachVertexBuffer()
{
    glBindBuffer(GL_ARRAY_BUFFER, 0);
}

bool
drawSubdivisions(WLEGLSurface* surface, InputDeviceConfiguration* conf)
{
    //printf("In drawSubdivisions\n");

    glClear(GL_COLOR_BUFFER_BIT);
    glUseProgram(gShaderObject.shaderProgramId);

    AttachVertexBuffer();

    const float touchAreaColor[] = {0.8, 0.8, 0.8, 1.0};
    const float sliderColor[] = {0.7, 0.9, 0.9, 1.0};
    const float buttonColor[] = {0.9, 0.4, 0.7, 1.0};
    const float* activeColor;

    list<Subdivision*> subdivisions = conf->getSubdivisions();

    unsigned int i = 0;
    for (list<Subdivision*>::reverse_iterator iter = subdivisions.rbegin();
         iter != subdivisions.rend();
         iter++)
    {
      //  printf("Drawing subdivision '%s'\n", (*iter)->getName().c_str());
        switch ((*iter)->getType())
        {
        case Subdivision::TOUCH:
        //    printf("It is a TouchArea subdivision\n");
            activeColor = touchAreaColor;
            break;
        case Subdivision::BUTTON:
        //    printf("It is a Button subdivision\n");
            activeColor = sliderColor;
            break;
        case Subdivision::SLIDER:
        //    printf("It is a Slider subdivision\n");
            activeColor = buttonColor;
            break;
        default:
            fprintf(stderr, "Subdivision with id %d is of an unknown type", (*iter)->getId());
            return false;
            break;
        }

        glUniformMatrix4fv(gShaderObject.matrixLocation, 1, GL_FALSE, world_mtx);
        glUniform4fv(gShaderObject.colorLocation, 1, activeColor);
        glDrawArrays(GL_TRIANGLE_FAN, i * 4, 4);
        glFinish();

        i++;
    }

    DetachVertexBuffer();

    eglSwapBuffers(surface->GetEGLDisplay(), surface->GetEGLSurface());

    return true;
}

/*Input listeners*/
const struct wl_pointer_listener PointerListener = {
    PointerHandleEnter,
    PointerHandleLeave,
    PointerHandleMotion,
    PointerHandleButton,
    PointerHandleAxis
};

const struct wl_keyboard_listener KeyboardListener = {
    KeyboardHandleKeymap,
    KeyboardHandleEnter,
    KeyboardHandleLeave,
    KeyboardHandleKey,
    KeyboardHandleModifiers,
};

const struct wl_touch_listener TouchListener = {
    TouchHandleDown,
    TouchHandleUp,
    TouchHandleMotion,
    TouchHandleFrame,
    TouchHandleCancel,
};

void
PointerHandleEnter(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                   struct wl_surface* wlSurface, wl_fixed_t sx, wl_fixed_t sy)
{
    WL_UNUSED(data);
    WL_UNUSED(wlPointer);
    WL_UNUSED(serial);
    WL_UNUSED(wlSurface);
    WL_UNUSED(sx);
    WL_UNUSED(sy);
    printf("ENTER EGLWLSubdivision PointerHandleEnter: x(%d), y(%d)\n", sx, sy);
}

void
PointerHandleLeave(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                   struct wl_surface* wlSurface)
{
    WL_UNUSED(data);
    WL_UNUSED(wlPointer);
    WL_UNUSED(serial);
    WL_UNUSED(wlSurface);
    printf("ENTER EGLWLSubdivision PointerHandleLeave: serial(%d)\n", serial);
}

void
PointerHandleMotion(void* data, struct wl_pointer* wlPointer, uint32_t time,
                    wl_fixed_t sx, wl_fixed_t sy)
{
    WL_UNUSED(data);
    WL_UNUSED(wlPointer);
    WL_UNUSED(time);

    printf("ENTER EGLWLSubdivision PointerHandleMotion: x(%d), y(%d)\n",
           (int)wl_fixed_to_double(sx),
           (int)wl_fixed_to_double(sy));
    gNeedRedraw = 1;
}

void
PointerHandleButton(void* data, struct wl_pointer* wlPointer, uint32_t serial,
                    uint32_t time, uint32_t button, uint32_t state)
{
    WL_UNUSED(data);
    WL_UNUSED(wlPointer);
    WL_UNUSED(serial);
    WL_UNUSED(time);
    WL_UNUSED(button);
    WL_UNUSED(state);
    printf("ENTER EGLWLSubdivision PointerHandleButton: button(%d), state(%d)\n", button, state);
}

void
PointerHandleAxis(void* data, struct wl_pointer* wlPointer, uint32_t time,
                  uint32_t axis, wl_fixed_t value)
{
    WL_UNUSED(data);
    WL_UNUSED(wlPointer);
    WL_UNUSED(time);
    WL_UNUSED(axis);
    WL_UNUSED(value);
    printf("ENTER EGLWLSubdivision PointerHandleAxis: axis(%d), value(%d)\n", axis, wl_fixed_to_int(value));
}

//////////////////////////////////////////////////////////////////////////////

void
KeyboardHandleKeymap(void* data, struct wl_keyboard* keyboard,
                     uint32_t format, int fd, uint32_t size)
{
    WL_UNUSED(data);
    WL_UNUSED(keyboard);
    WL_UNUSED(format);
    WL_UNUSED(fd);
    WL_UNUSED(size);
    printf("ENTER EGLWLSubdivision KeyboardHandleKeymap: format(%d), fd(%d), size(%d)\n",
        format, fd, size);
}

void
KeyboardHandleEnter(void* data, struct wl_keyboard* keyboard, uint32_t serial,
                    struct wl_surface* surface, struct wl_array* keys)
{
    WL_UNUSED(data);
    WL_UNUSED(keyboard);
    WL_UNUSED(serial);
    WL_UNUSED(surface);
    WL_UNUSED(keys);
    printf("ENTER EGLWLSubdivision KeyboardHandleEnter: serial(%d), surface(%p)\n",
        serial, surface);
}

void
KeyboardHandleLeave(void* data, struct wl_keyboard* keyboard, uint32_t serial,
                    struct wl_surface* surface)
{
    WL_UNUSED(data);
    WL_UNUSED(keyboard);
    WL_UNUSED(serial);
    WL_UNUSED(surface);
    printf("ENTER EGLWLSubdivision KeyboardHandleLeave: serial(%d), surface(%p)\n",
        serial, surface);
}

void
KeyboardHandleKey(void* data, struct wl_keyboard* keyboard, uint32_t serial,
                  uint32_t time, uint32_t key, uint32_t state_w)
{
    WL_UNUSED(data);
    WL_UNUSED(keyboard);
    WL_UNUSED(serial);
    WL_UNUSED(time);
    WL_UNUSED(key);
    WL_UNUSED(state_w);
    printf("ENTER EGLWLSubdivision KeyboardHandleKey: serial(%d), time(%d), key(%d), state_w(%d)\n",
        serial, time, key, state_w);
}

void
KeyboardHandleModifiers(void* data, struct wl_keyboard* keyboard, uint32_t serial,
                        uint32_t mods_depressed, uint32_t mods_latched,
                        uint32_t mods_locked, uint32_t group)
{
    WL_UNUSED(data);
    WL_UNUSED(keyboard);
    WL_UNUSED(serial);
    WL_UNUSED(mods_depressed);
    WL_UNUSED(mods_latched);
    WL_UNUSED(mods_locked);
    WL_UNUSED(group);
    printf("ENTER EGLWLSubdivision KeyboardHandleModifiers: serial(%d), mods_depressed(%d)\n"
          "                               mods_latched(%d), mods_locked(%d)\n"
           "                               group(%d)\n",
        serial, mods_depressed, mods_latched, mods_locked, group);
}

//////////////////////////////////////////////////////////////////////////////

void
TouchHandleDown(void* data, struct wl_touch* touch, uint32_t serial, uint32_t time,
                struct wl_surface* surface, int32_t id, wl_fixed_t xw, wl_fixed_t yw)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
    WL_UNUSED(serial);
    WL_UNUSED(time);
    WL_UNUSED(surface);
    WL_UNUSED(id);
    WL_UNUSED(xw);
    WL_UNUSED(yw);

    gNeedRedraw = 1;
    printf("ENTER EGLWLSubdivision TouchHandleDown id: %d, x: %d,y: %d \n",
            id,
            (int)wl_fixed_to_double(xw),
            (int)wl_fixed_to_double(yw));
}

void
TouchHandleUp(void* data, struct wl_touch* touch, uint32_t serial, uint32_t time, int32_t id)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
    WL_UNUSED(serial);
    WL_UNUSED(time);
    WL_UNUSED(id);
    printf("ENTER EGLWLSubdivision TouchHandleUp\n");
}

void
TouchHandleMotion(void* data, struct wl_touch* touch, uint32_t time, int32_t id,
                  wl_fixed_t xw, wl_fixed_t yw)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
    WL_UNUSED(time);
    WL_UNUSED(id);
    WL_UNUSED(xw);
    WL_UNUSED(yw);

    printf("ENTER EGLWLSubdivision TouchHandleMotion id: %d, x: %d,y: %d \n",
            id,
            (int)wl_fixed_to_double(xw),
            (int)wl_fixed_to_double(yw));
    gNeedRedraw = 1;
}

void
TouchHandleFrame(void* data, struct wl_touch* touch)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
}

void
TouchHandleCancel(void* data, struct wl_touch* touch)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
}
